<?php

defined('BASEPATH') || exit('No direct script access allowed');

require_once __DIR__ . '/../WhatsbotapiController.php';

use Whatsbotapi\WhatsbotapiController;

class Templatebots extends WhatsbotapiController
{
    public function __construct()
    {
        parent::__construct();

        $this->load->model('whatsbotapi_model');
        $this->load->library('app_modules');
        $this->load->helper('whatsbotapi');

        if ($this->app_modules->is_inactive('whatsbot')) {
            $this->response(isWhatsbotNotActive()['response'], isWhatsbotNotActive()['response_code']);
        }

        register_language_files('whatsbotapi');
        load_client_language();

        if (!defined('WHATSBOT_MODULE_UPLOAD_FOLDER')) {
            define('WHATSBOT_MODULE_UPLOAD_FOLDER', 'uploads/whatsbot');
        }


        if (isWbAuthorized()) {
            $this->response(isWbAuthorized()['response'], isWbAuthorized()['response_code']);
        }
    }

    /**
     * @api {get} /whatsbotapi/v1/templatebots List All Template Bots
     *
     * @apiVersion 1.0.0
     *
     * @apiName Get All Template Bots
     *
     * @apiGroup TemplateBots
     *
     * @apiHeader {String} Authorization <span class="btn btn-xs btn-danger">Required</span> Basic Access Authentication token.
     *
     * @apiSampleRequest /whatsbotapi/v1/templatebots
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {Array}   data    Template bots.
     * @apiSuccess {String}  message Success message.
     *
     * @apiSuccessExample Success-Response:
     *     HTTP/1.1 200 OK
     *     {
     *          "status": true,
     *          "data": [
     *              {
     *                  "id": "1",
     *                  "name": "uraj",
     *                  "rel_type": "leads",
     *                  "template_id": "1",
     *                  "scheduled_send_time": null,
     *                  "send_now": "0",
     *                  "header_params": "{\"1\":{\"value\":\"\"}}",
     *                  "body_params": "{\"1\":{\"value\":\"\"},\"2\":{\"value\":\"\"},\"3\":{\"value\":\"\"}}",
     *                  "footer_params": "[]",
     *                  "filename": null,
     *                  "pause_campaign": "0",
     *                  "select_all": "0",
     *                  "trigger": "how",
     *                  "bot_type": "1",
     *                  "is_bot_active": "1",
     *                  "is_bot": "1",
     *                  "is_sent": "0",
     *                  "created_at": "2024-09-21 12:18:01",
     *                  "sending_count": "0",
     *                  "template_name": "testing_template",
     *                  "tmp_id": "559989593050190",
     *                  "header_params_count": "1",
     *                  "body_params_count": "3",
     *                  "footer_params_count": "0"
     *              }
     *          ],
     *          "message": "Data retrieved successfully"
     *     }
     *
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message No data found.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 404 Not Found
     *     {
     *       "status": false,
     *       "message": "No data found"
     *     }
     */


    /**
     * @api {get} /whatsbotapi/v1/templatebots/id/1 Template Bot information
     *
     * @apiVersion 1.0.0
     *
     * @apiName Request Template Bot information
     *
     * @apiGroup TemplateBots
     *
     * @apiHeader {String} Authorization <span class="btn btn-xs btn-danger">Required</span> Basic Access Authentication token.
     *
     * @apiSampleRequest off
     *
     * @apiParam {Number} id <span class="btn btn-xs btn-danger">Required</span> Template Bots unique ID.
     * @apiParam {String} group=TemplateBots <span class="btn btn-xs btn-danger">Required</span> Group Name
     *
     * @apiBody {String} content <span class="btn btn-xs btn-danger">Required</span> Comment
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {String}  message Success message.
     *
     * @apiSuccessExample Success-Response:
     *     HTTP/1.1 200 OK
     *     {
     *         "status": true,
     *         "message": "message": "Data retrived successfully"
     *     }
     *
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message Error message.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 403 Forbidden
     *     {
     *       "status": false,
     *       "message": "No data found"
     *     }
     */

    public function templatebots_get()
    {
        if (!empty($this->get()) && !in_array('id', array_keys($this->get()))) {
            $this->response(['message' => _l('something_went_wrong')], 500);
        }

        if (isset($this->get()['id']) && !is_numeric($this->get('id'))) {
            $this->response(['message' => _l('id_must_be_numeric')], 500);
        }

        $templateID   = $this->get('id');
        $templateData = $this->whatsbotapi_model->getTable('wtc_templatebots', $templateID);

        $this->response($templateData['response'], $templateData['response_code']);
    }

    public function templatebots_post()
    {
        $requiredData = [
            'name' => '',
            'rel_type' => '',
            'template_id' => '',
            'bot_type' => '',
            'trigger' => ''
        ];

        $postData = $this->post();
        $postData = array_merge($requiredData, $postData);

        $this->load->library('form_validation');

        $this->form_validation->set_data($postData);

        $this->form_validation->set_rules('name', _l('bot_name'), 'required');
        $this->form_validation->set_rules('rel_type', _l('rel_type'), 'required|in_list[leads,contacts]');
        $this->form_validation->set_rules('template_id', _l('template'), 'required|integer');
        $this->form_validation->set_rules('bot_type', _l('bot_type'), 'required|numeric|in_list[1,2,3,4]');

        if ($this->post('bot_type') != 3) {
            $this->form_validation->set_rules('trigger', _l('trigger'), 'required');
        }

        if ($this->form_validation->run() === FALSE) {
            $this->response(['message' => strip_tags(validation_errors())], 422);
        }

        $this->load->model('whatsbot/campaigns_model');

        $res = $this->bots_model->saveTemplateBot($postData);

        if ($res['type'] == 'success') {
            wb_handle_campaign_upload($res['temp_id'], 'template');
        }

        ($res['type'] == 'success') ? $this->response(['message' => _l('bot_create_successfully')], 200) : $this->response(['message' => _l('something_went_wrong')], 500);
    }

    public function templatebots_delete()
    {
        if (!empty($this->get()) && !in_array('id', array_keys($this->get()))) {
            $this->response(['message' => _l('something_went_wrong')], 500);
        }

        if (isset($this->get()['id']) && !is_numeric($this->get('id'))) {
            $this->response(['message' => _l('id_must_be_numeric')], 500);
        }

        $bot_id = $this->get('id');
        $res = $this->bots_model->deleteMessageBot('template', $bot_id);

        ($res['status']) ? $this->response(['message' => _l('bot_deleted_successfully')], 200) : $this->response(['message' => _l('no_record_found')], 500);
    }
}
