<?php

defined('BASEPATH') || exit('No direct script access allowed');

require_once __DIR__ . '/../WhatsbotapiController.php';
require_once __DIR__ . '/../../../whatsbot/vendor/autoload.php';

use ParagonIE\Sodium\Core\Curve25519\Ge\Precomp;
use Whatsbotapi\WhatsbotapiController;

class Campaigns extends WhatsbotapiController
{
    public function __construct()
    {
        parent::__construct();

        $this->load->model(['whatsbotapi_model', 'whatsbot/campaigns_model', 'whatsbot/whatsbot_model']);
        $this->load->library('app_modules');
        $this->load->library('form_validation');
        $this->load->helper('whatsbotapi');

        if ($this->app_modules->is_inactive('whatsbot')) {
            $this->response(isWhatsbotNotActive()['response'], isWhatsbotNotActive()['response_code']);
        }
        
        if (!defined('WHATSBOT_MODULE_UPLOAD_FOLDER')) {
            define('WHATSBOT_MODULE_UPLOAD_FOLDER', 'uploads/whatsbot');
        }

        register_language_files('whatsbotapi');
        load_client_language();

        if (isWbAuthorized()) {
            $this->response(isWbAuthorized()['response'], isWbAuthorized()['response_code']);
        }
    }

    /**
     * @api {get} yourdomain.com/whatsbotapi/v1/campaigns Get all Campaigns
     *
     * @apiVersion 1.0.0
     *
     * @apiName Get all Campaigns
     *
     * @apiGroup Campaigns
     *
     * @apiHeader {String} Authorization <span class="btn btn-xs btn-danger">Required</span> Basic Access Authentication token.
     *
     * @apiSampleRequest /whatsbotapi/v1/campaigns
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {Array}   data    Campaigns
     * @apiSuccess {String}  message Success message.
     *
     * @apiSuccessExample Success-Response:
     * HTTP/1.1 200 OK
     * {
     * "status": true,
     * "data": [
     *     {
     *         "id": "1",
     *         "name": "Test campaign",
     *         "rel_type": "contacts",
     *         "template_id": "1",
     *         "scheduled_send_time": null,
     *         "send_now": "0",
     *         "header_params": "{\"1\":{\"value\":\"Hello\"}}",
     *         "body_params": "{\"1\":{\"value\":\"variable\"},\"2\":{\"value\":\"\"},\"3\":{\"value\":\"\"}}",
     *         "footer_params": "[]",
     *         "filename": null,
     *         "pause_campaign": "0",
     *         "select_all": "0",
     *         "trigger": null,
     *         "bot_type": "0",
     *         "is_bot_active": "1",
     *         "is_bot": "0",
     *         "is_sent": "0",
     *         "created_at": "2024-09-21 12:14:47",
     *         "sending_count": "0",
     *         "template_name": "testing_template",
     *         "tmp_id": "559989593050190",
     *         "header_params_count": "1",
     *         "body_params_count": "3",
     *         "footer_params_count": "0",
     *         "rel_ids": "[1]"
     *     },
     *     {
     *         "id": "2",
     *         "name": "Another test campaign ",
     *         "rel_type": "contacts",
     *         "template_id": "2",
     *         "scheduled_send_time": "2024-09-21 12:15:58",
     *         "send_now": "1",
     *         "header_params": "[]",
     *         "body_params": "[]",
     *         "footer_params": "[]",
     *         "filename": null,
     *         "pause_campaign": "0",
     *         "select_all": "0",
     *         "trigger": null,
     *         "bot_type": "0",
     *         "is_bot_active": "1",
     *         "is_bot": "0",
     *         "is_sent": "1",
     *         "created_at": "2024-09-21 12:15:58",
     *         "sending_count": "1",
     *         "template_name": "hello_world",
     *         "tmp_id": "958186559334198",
     *         "header_params_count": "0",
     *         "body_params_count": "0",
     *         "footer_params_count": "0",
     *         "rel_ids": "[1]"
     *     }
     * ],
     * "message": "Data retrived successfully"
     * }
     *
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message No data found.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 404 Not Found
     *     {
     *       "status": false,
     *       "message": "No data found"
     *     }
     */

    /**
     * @api {get} /whatsbotapi/v1/campaigns/id/:id Request Campaign information
     *
     * @apiVersion 1.0.0
     *
     * @apiName Request Campaign information
     *
     * @apiGroup Campaigns
     *
     * @apiHeader {String} Authorization <span class="btn btn-xs btn-danger">Required</span> Basic Access Authentication token.
     *
     * @apiParam {Number} id <span class="btn btn-xs btn-danger">Required</span> Campaign unique ID.
     * 
     * @apiSampleRequest /whatsbotapi/v1/campaigns/id/1
     *
     * @apiSuccess {Boolean} status  Response status.
     * @apiSuccess {Object}   data    Campaign
     * @apiSuccess {String}  message Success message.
     *
     * @apiSuccessExample Success-Response:
     * HTTP/1.1 200 OK
     * {
     * {
     *  "status": true,
     *  "data": {
     *      "id": "1",
     *      "name": "Test campaign",
     *      "rel_type": "contacts",
     *      "template_id": "1",
     *      "scheduled_send_time": null,
     *      "send_now": "0",
     *      "header_params": "{\"1\":{\"value\":\"Hello\"}}",
     *      "body_params": "{\"1\":{\"value\":\"variable\"},\"2\":{\"value\":\"\"},\"3\":{\"value\":\"\"}}",
     *      "footer_params": "[]",
     *      "filename": null,
     *      "pause_campaign": "0",
     *      "select_all": "0",
     *      "trigger": null,
     *      "bot_type": "0",
     *      "is_bot_active": "1",
     *      "is_bot": "0",
     *      "is_sent": "0",
     *      "created_at": "2024-09-21 12:14:47",
     *      "sending_count": "0",
     *      "template_name": "testing_template",
     *      "tmp_id": "559989593050190",
     *      "header_params_count": "1",
     *      "body_params_count": "3",
     *      "footer_params_count": "0",
     *      "rel_ids": "[1]"
     *  },
     *  "message": "Data retrived successfully"
     * }
     *
     * @apiError {Boolean} status  Response status.
     * @apiError {String}  message No data found.
     *
     * @apiErrorExample Error-Response:
     *     HTTP/1.1 404 Not Found
     *     {
     *       "status": false,
     *       "message": "No data found"
     *     }
     */
    public function campaigns_get()
    {
        if (!empty($this->get()) && !in_array('id', array_keys($this->get()))) {
            $this->response(['message' => _l('something_went_wrong')], 500);
        }

        if (isset($this->get()['id']) && !is_numeric($this->get('id'))) {
            $this->response(['message' => _l('id_must_be_numeric')], 500);
        }

        $campaignID   = $this->get('id');
        $campaignData = $this->whatsbotapi_model->getTable('wtc_campaigns', $campaignID);

        $this->response($campaignData['response'], $campaignData['response_code']);
    }

    public function campaigns_post()
    {
        $requiredData = [
            'name' => '',
            'rel_type' => '',
            'template_id' => ''
        ];

        $postData = $this->post();
        $postData = array_merge($requiredData, $postData);

        ($postData['rel_type'] == 'leads') ? $requiredData['lead_ids'] = '' : ($postData['rel_type'] == 'contacts' ?  $requiredData['contact_ids'] = '' : '');

        (isset($postData['scheduled_send_time']) && !empty($postData['scheduled_send_time'])) ? $requiredData['scheduled_send_time'] = '' : '';

        $postData = array_merge($requiredData, $postData);

        $this->form_validation->set_data($postData);
        $this->form_validation->set_rules('name', _l('campaign_name'), 'required');
        $this->form_validation->set_rules('rel_type', _l('rel_type'), 'required|in_list[leads,contacts]');
        $this->form_validation->set_rules('template_id', _l('template'), 'required|integer');

        ($postData['rel_type'] == 'leads') ? $this->form_validation->set_rules('lead_ids', _l('lead_ids'), 'required') : ($postData['rel_type'] == 'contacts' ? $this->form_validation->set_rules('contact_ids', _l('contact_ids'), 'required') : '');

        if (isset($postData['scheduled_send_time']) && !empty($postData['scheduled_send_time'])) {
            $this->form_validation->set_rules('scheduled_send_time', _l('contact_ids'), 'required|callback_validate_date');
        } else {
            $postData['send_now'] = 1;
        }

        if ($this->form_validation->run() === FALSE) {
            $this->response(['message' => strip_tags(validation_errors())], 422);
        }

        ($postData['rel_type'] == 'leads') ? $postData['lead_ids'] = json_decode($postData['lead_ids']) : ($postData['rel_type'] == 'contacts' ? $postData['contact_ids'] = json_decode($postData['contact_ids'])  : '');

        $res = $this->campaigns_model->save($postData);

        if ($res['type'] == 'success') {
            wb_handle_campaign_upload($res['campaign_id'], 'template');
        }

        ($res['type'] == 'success') ? $this->response(['message' => $res['message']], 200) : $this->response(['message' => _l('something_went_wrong')], 500);
    }

    // Action to validate date input that must be in past
    public function validate_date($schedule_date)
    {
        $today = date("Y-m-d H:i");

        if (strtotime($schedule_date) < strtotime($today)) {
            $this->form_validation->set_message('validate_date', 'Schedule date should not be in the past');
            return FALSE;
        } else {
            return TRUE;
        }
    }

    public function campaigns_delete()
    {
        if (empty($this->get()) && !in_array('id', array_keys($this->get()))) {
            $this->response(['message' => _l('provide_id_delete_record')], 500);
        }

        if (isset($this->get()['id']) && !is_numeric($this->get('id'))) {
            $this->response(['message' => _l('id_must_be_numeric')], 500);
        }

        $id = $this->get('id');

        $res = $this->campaigns_model->delete($id);

        ($res['status']) ? $this->response(['message' => _l('campaign_deleted_successfully')], 200) : $this->response(['message' => _l('no_record_found')], 500);
    }
}
